/*
*********************************************************************************************************
*                                                ALLD.cpp
*
* Description : This is the main file. It performs the advanced detection of lane lines on an input image
				(or video) passed from the command line saving the final result to a output file.
*
*
*
*********************************************************************************************************
*/


/*
************************************************************************************************************************
*                                                 INCLUDE LIBRARIES FILES
************************************************************************************************************************
*/

#include <opencv2/opencv.hpp>
#include <iostream>
#include <stdlib.h>
#include <string>
#include <filesystem>
#include <math.h>
#include <cstdlib> 
#include <fstream>
#include <sstream>
#include <time.h>
#include <algorithm>
#include <iterator>
#include <iomanip>
#include <limits>
#include <opencv2/videoio.hpp>
#include <opencv2/core/ocl.hpp>
#include <opencv2/video.hpp>
#include "ALLD_lib.h"

/*
************************************************************************************************************************
*                                                 FUNCTION PROTOTYPES
************************************************************************************************************************
*/

static void help();
void mouseOnClick(int event, int x, int y, int flags, void* param);

/*
************************************************************************************************************************
*                                                 SET ARGUMENT VARIABLES
************************************************************************************************************************
*/

const char* keys =
"{help h usage ?  |      | Usage examples: \n\t\t ALLD.exe -i=image.jpg (or -i=video.avi) }"
"{input i         |<none>| input image or video				}"
;


/*
************************************************************************************************************************
*                                                 SET NAMESPACES
************************************************************************************************************************
*/

using namespace std;
using namespace cv;

namespace fs = std::experimental::filesystem;

/*
************************************************************************************************************************
*                                                 GLOBAL VARIABLES
************************************************************************************************************************
*/

int bad = 0; // number of bad frames (not correctly detected)

/*
*********************************************************************************************************
*                                            HELP FUNCTION
*
* Description : This is a helper function.
*
*
*********************************************************************************************************
*/


static void help()
{
	cout
		<< "------------------------------------------------------------------------------" << endl
		<< "This program apply to an image or video the Advanced Lane Lines Detection Algorithm:" << endl
		<< "Usage: ALLD.exe -i=image.jpg (or -i=video.avi)	   " << endl
		<< "------------------------------------------------------------------------------" << endl
		<< endl;
}

/*
*********************************************************************************************************
*                                            MOUSEONCLICK FUNCTION
*
* Description : This function captures the x and y coordinates of the left mouse click displaying them on the console.
				It can be used to search for points in the region of interest in an image
*
*
*********************************************************************************************************
*/

// use this function to set the ROI of the image
void mouseOnClick(int event, int x, int y, int flags, void* param) {
	if (event == EVENT_LBUTTONDOWN) {
		std::cout << "Coordinates: " << "( x: " << x << ", y: " << y << " )" << endl;
	}
	return;
}

/*
*********************************************************************************************************
*                                            MAIN FUNCTION
*
* Description : This is the main function used to apply the ALLD algorithm
*
*
*********************************************************************************************************
*/


int main(int argc, char** argv) {

	cv::CommandLineParser parser(argc, argv, keys);
	parser.about("Use this script to run advanced lane line detection using OpenCV.");

	if (argc == 1) {
		cout << "Wrong argument, press -h for more information..." << endl;
		return -1;
	}
	if (parser.has("help"))
	{
		cout << "help message:" << endl;
		parser.printMessage();
		return 0;
	}

	if (argc == 2) {
		
		std::string path;
		std::string path_out;

		if (parser.has("input")) {
			path = parser.get<String>("input");
			std::cout << "input path is: " << path << std::endl;
		}
		else {
			std::cout << "Error reading the file..." << std::endl;
			return -1;
		}
		
		string ext = getFileExt(path);
		std::cout << "The input file extension is: \"" << ext << "\"\n";

		path_out = "Final_out";
		std::cout << "default output path is: " << path_out + "." + ext << std::endl;

		if (!parser.check())
		{
			parser.printErrors();
			return 0;
		}

		// check if the input extension belongs to the most common list of formats supported by OpenCV

		if (ext == "jpg" || ext == "png" || ext == "jpeg") {

			cout << "Loading the image..." << endl;

			Mat image = imread(path);

			// Color image consist of at least 3 layers: Red, Green and Blue.
			// Any pixel is a combination of the three values. Any color can be created by combining these 3 basic colors.
			// (255, 0, 0) represents 100% Red. 
			// (0, 255, 0) represents 100% Green.
			// (0, 0, 255) represents 100% Blue.

			if (image.empty()) //check for failure
			{
				std::cout << "Could not open or find the image!" << std::endl;
				std::system("pause"); //wait for any key press
				return -1;
			}
			std::cout << "Image is successfully loaded from path: " << path << std::endl;
			std::cout << "Size of the Original Image: ";
			std::cout << "Width: " << image.size().width << " x Height: " << image.size().height << std::endl;
			
			// Resizing the image into 1280 x 720 -> HD resolution
			std::cout << "Resizing the image into 1280x720 -> HD resolution...\n";

			Size size(1280, 720);	// destination image size
			Mat dst;				// destination image

			// Take the time
			double t = (double)getTickCount();
			
			if (image.size() != size) {
				change_Size(image, image, size);  // Resize the image if it is not in HD resolution
			}
		
			std::cout << "Done! " << std::endl << "Showing the Image...\n";
			std::cout << "Width: " << image.size().width << " x Height: " << image.size().height << endl;

			show_and_save("Original", image, ext, true, false);

			dst = image.clone();

			/* uncomment the following line in order activate the callback function
			   to get the x and y mouse left click position to set the ROI*/

			   // setMouseCallback("Original Image", mouseOnClick, &dst);

			cout << "Applying Advanced Lane Detection Algorithm to the image..." << endl;

			cv::Mat warped, Minv, M;	// Mat declarations for warped image, the perspective transformation matrix and its inverse
			cv::Mat combined;			// Mat declaration for the combined binary image
			cv::Mat hist;				// Mat declaration for the histogram
			int height = dst.rows;      // image height
			int width = dst.cols;		// image width

			// In order to correctly identify the lane lines a combination of color and gradient thresholds
			// is performed to generate a thresholded binary image
			combined_threshold(dst, combined, ext);

			// Applies a perspective transformation ("bird's-eye view") to a thresholded binary image
			binary_transform(dst, warped, M, Minv, combined, ext);

			// To detect the lane lines initial positions the histogram is computed
			get_histogram(warped, hist);

			cv::Mat window;
			vector<Point> leftPt, rightPt;
			vector<Point> leftFitPt, rightFitPt;
			vector<float> leftFit_x, rightFit_x;

			// Applies a sliding window method both for the left and the right lane line, in order to follow accurately the lane lines pixels,
			// returning  the struct of vector points which contain window area for left and right lines
			Points pt = sliding_window(warped, hist, window, leftPt, rightPt, leftFitPt, rightFitPt, leftFit_x, rightFit_x);
			
			// Calculate the Radius of curvature R in meters for each lane line
			float left_curverad = measure_curvature(leftFit_x, height);
			float right_curverad = measure_curvature(rightFit_x, height);
			// Take the difference and the average curvature between the two
			float curve_diff = abs(left_curverad - right_curverad);
			float curve_avg = (left_curverad + right_curverad) / 2;

			// Calculate the offset of the vehicle with respect to the center of the lane
			string direction;
			float deviation = measure_vehicle_offset(leftFit_x, rightFit_x, direction, width);

			stringstream curv;
			// Set font text
			int fontFace = FONT_HERSHEY_SIMPLEX;
			double fontScale = 1.5;
			int thickness = 3;
			cv::Point2f text(10, 60);

			// if the average curvature is high than a threshold the curve can be approximated as a straight line
			if (curve_avg > 3000) {
				curv << "Straight";
			}
			else {
				curv << "Average Curvature is: " << curve_avg << " m";
			}

			stringstream off;
			off << "Vehicle is: " << abs(deviation) << " m " << direction << " of center";
			cv::Point2f text_org(10, 120);

			// Check if the frame lane lines have been detected correctly
			if (leftFitPt.size() == 0 || rightFitPt.size() == 0) {
				bad += 1;
				cout << "frame non correctly detected! " << endl;
			}

			// Draw the right and left lane lines in the output image
			cv::Mat finale;
			draw_lane_lines(dst, finale, Minv, leftFitPt, rightFitPt, pt);

			// Clear the vector points
			clear_Fit_Point(leftFitPt, rightFitPt, pt);

			// Add text with on-screen information
			// Lines curvature information
			putText(finale, curv.str(), text, fontFace, fontScale, Scalar(0, 0, 255, 255), thickness, 8);
			// Vehicle's offset information
			putText(finale, off.str(), text_org, fontFace, fontScale, Scalar(0, 0, 255, 255), thickness, 8);

			t = ((double)getTickCount() - t) / getTickFrequency();
			std::cout << "Processing time: " << t << " s" << std::endl;

			// Save the final image
			show_and_save(path_out, finale, ext, true, true);
			
			cout << "Finish. Exiting...\n";
			system("pause");

		}

		// check if the input extension belongs to the most common list of formats supported by OpenCV

		else if (ext == "avi" || ext == "mp4") {

			cout << "Processing the video: " << path << endl;
			
			// Create a VideoCapture object and open the input file
			cv::VideoCapture cap(path);
			// Check if the video is opened successfully
			if (cap.isOpened() == false) { 
				std::cout << "Error opening video file" << std::endl;
				cin.get();
				exit(EXIT_FAILURE);
			}

			// Take some informations from the video frames
			float fps = static_cast<float>(cap.get(CAP_PROP_FPS));
			//std::cout << "Frames per seconds : " << fps << std::endl;
			int frame_width = static_cast<int>(cap.get(CAP_PROP_FRAME_WIDTH));		// get the width of frames of the video
			int frame_height = static_cast<int>(cap.get(CAP_PROP_FRAME_HEIGHT));	// get the height of frames of the video
			int frame_count = static_cast<int>(cap.get(CAP_PROP_FRAME_COUNT));		// get the total frame number of the video
			
			// Take the video duration
			float duration = frame_count / fps;
			bool flag = true;

			std::cout << "Size of frames (width x height):" << frame_width << " x " << frame_height << endl;
			std::cout << "Total number of frame detected: " << frame_count << endl;
			std::cout << "Total fps: " << fps << endl;
			std::cout << "Video Duration: " << std::fixed << std::setprecision(3) << duration << " seconds" << std::endl;

			// Final output video size
			cv::Size frame_size(1280, 720);
			int frames_per_second = 20;  // framerate of the output video

			if (frame_size != cv::Size(frame_width, frame_height)) {
				cout << "The output will be resized to 1280x720 HD resolution" << endl;
			}

			// Initialize the VideoWriter object and define the fourcc.
			// fourcc stands for 4-character code of the codec used to compress the video.
			// string outputFile = "out.mp4";
			VideoWriter video(path_out + "." + ext, VideoWriter::fourcc('M', 'J', 'P', 'G'), frames_per_second, frame_size, true);

			// If the VideoWriter object is not initialized successfully, exit the program
			if (video.isOpened() == false)
			{
				std::cout << "Unable to save the video to a file" << endl;
				cin.get(); //wait for any key press
				return -1;
			}

			cv::Mat warped, Minv, M;	// Mat declarations for warped image, the perspective transformation matrix and its inverse
			cv::Mat combined;			// Mat declaration for the combined binary image
			cv::Mat hist;				// Mat declaration for the histogram
			int nframe = 0;				// count the number of frames
			
			// Take the time
			double t = (double)getTickCount();

			// When the VideoCapture object is created, it is possible to capture video frame by frame
			while (true) {

				cv::Mat frame;

				bool isSuccess = cap.read(frame); // read a new frame from the video camera
				// Breaking the while loop if frames cannot be read from the video file
				if (isSuccess == false)
				{
					cout << "Video disconnected" << endl;
					break;
				}
				
				if (frame.empty()) break;	// If the frame is empty, break immediately

				// Resizing the image to 1280 x 720 HD resolution if needed
				if (frame_size != cv::Size(frame_width, frame_height)) {
					change_Size(frame, frame, frame_size);
				}

				// In order to correctly identify the lane lines a combination of color and gradient thresholds
				// is performed to generate a thresholded binary image
				combined_threshold(frame, combined, "jpg");

				// Applies a perspective transformation ("bird's-eye view") to a thresholded binary image
				binary_transform(frame, warped, M, Minv, combined, "jpg");

				// To detect the lane lines initial positions the histogram is computed
				get_histogram(warped, hist);

				cv::Mat window;
				vector<Point> leftPt, rightPt;
				vector<Point> leftFitPt, rightFitPt;
				vector<float> leftFit_x, rightFit_x;
								
				// Applies a sliding window method both for the left and the right lane line, in order to follow accurately the lane lines pixels,
				// returning  the struct of vector points which contain window area for left and right lines
				Points pt = sliding_window(warped, hist, window, leftPt, rightPt, leftFitPt, rightFitPt, leftFit_x, rightFit_x);
				
				// Calculate the Radius of curvature R in meters for each lane line
				float left_curverad = measure_curvature(leftFit_x, frame_size.height);
				float right_curverad = measure_curvature(rightFit_x, frame_size.height);
				// Take the difference and the average curvature between the two
				float curve_diff = abs(left_curverad - right_curverad);
				float curve_avg = (left_curverad + right_curverad) / 2;

				// Calculate the offset of the vehicle
				string direction;
				float deviation = measure_vehicle_offset(leftFit_x, rightFit_x, direction, frame_size.width);
				
				stringstream curv;
				int fontFace = FONT_HERSHEY_SIMPLEX;
				double fontScale = 1.5;
				int thickness = 3;
				cv::Point2f text(10, 60);
				// if the average curvature is high than a threshold the curve can be approximated as a straight line
				if (curve_avg > 3000) {
					curv << "Straight";
				}
				else {
					curv << "Average Curvature is: " << curve_avg << " m";
				}

				stringstream off;
				off << "Vehicle is: " << abs(deviation) << " m " << direction << " of center";
				cv::Point2f text_org(10, 120);
				
				// Sanity check : whether the lines are roughly parallel and have similar curvature
			
				float curve_threshold = 15000.f;
				
				// cout << "left_curverad: " << left_curverad << endl;
 				// cout << "right_curverad: " << right_curverad << endl;
 				// cout << "curve_avg: " << curve_avg << endl;
 				// cout << "curve_diff: " << curve_diff << endl;

				if (nframe > 0) {
					if (curve_diff > curve_threshold) {
						bad += 1;
						cout << "frame: " << nframe << " non correctly detected! " << endl;
					}
				}

				// Draw the right and left lane lines in the output image
				cv::Mat finale;
				draw_lane_lines(frame, finale, Minv, leftFitPt, rightFitPt, pt);

				// Clear the vector points
				clear_Fit_Point(leftFitPt, rightFitPt, pt);

				// Add text with on-screen information
				// Lines curvature information
				putText(finale, curv.str(), text, fontFace, fontScale, Scalar(0, 0, 255, 255), thickness, 8);
				// Vehicle's offset information
				putText(finale, off.str(), text_org, fontFace, fontScale, Scalar(0, 0, 255, 255), thickness, 8);
				
				nframe++;	// count the total number of frames
				std::cout << "Processing frame n.: " << nframe << std::endl;

				if (flag == true) {
					flag = false;
					// To obtain a true value, disable any intermediate display
					t = ((double)getTickCount() - t) / getTickFrequency();
					std::cout << "Time to process one frame in seconds: " << t << " s" << std::endl;
					std::cout << "Estimated total time to finish: " << t* frame_count << " s" << endl;
				}

				// Write the video frame to the output file
				video.write(finale);
				if (cv::waitKey(10) == 27)
				{
					std::cout << "Esc key is pressed by the user! Stopping the video..." << endl;
					break;
				}
			}
			cout << "Number of bad frame detected: " << bad << endl;
			cout << "Number of good frame detected: " << nframe-bad << endl;
			cout << "Total number of Frame Processed: " << nframe << endl;
			cout << "Finished writing..." << endl;
			cout << "Video is successfully saved to a file: " << path_out + "." + ext << endl;

			// Flush and close the video file
			cap.release();
			video.release();
			system("pause");
			return 0;
		}
		else {
			std::cout << "Format extension not supported! Exit...\n";
			system("pause");
			return -1;
		}
	}
	else {
		std::cout << "The number of arguments is wrong. Exit...\n";
		system("PAUSE");
		return -1;
	}
	return 0;
}